﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/finspace/Finspace_EXPORTS.h>
#include <aws/finspace/model/KxScalingGroupStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace finspace {
namespace Model {

/**
 * <p> A structure for storing metadata of scaling group.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/KxScalingGroup">AWS
 * API Reference</a></p>
 */
class KxScalingGroup {
 public:
  AWS_FINSPACE_API KxScalingGroup() = default;
  AWS_FINSPACE_API KxScalingGroup(Aws::Utils::Json::JsonView jsonValue);
  AWS_FINSPACE_API KxScalingGroup& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FINSPACE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A unique identifier for the kdb scaling group. </p>
   */
  inline const Aws::String& GetScalingGroupName() const { return m_scalingGroupName; }
  inline bool ScalingGroupNameHasBeenSet() const { return m_scalingGroupNameHasBeenSet; }
  template <typename ScalingGroupNameT = Aws::String>
  void SetScalingGroupName(ScalingGroupNameT&& value) {
    m_scalingGroupNameHasBeenSet = true;
    m_scalingGroupName = std::forward<ScalingGroupNameT>(value);
  }
  template <typename ScalingGroupNameT = Aws::String>
  KxScalingGroup& WithScalingGroupName(ScalingGroupNameT&& value) {
    SetScalingGroupName(std::forward<ScalingGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The memory and CPU capabilities of the scaling group host on which FinSpace
   * Managed kdb clusters will be placed.</p> <p>You can add one of the following
   * values:</p> <ul> <li> <p> <code>kx.sg.large</code> – The host type with a
   * configuration of 16 GiB memory and 2 vCPUs.</p> </li> <li> <p>
   * <code>kx.sg.xlarge</code> – The host type with a configuration of 32 GiB memory
   * and 4 vCPUs.</p> </li> <li> <p> <code>kx.sg.2xlarge</code> – The host type with
   * a configuration of 64 GiB memory and 8 vCPUs.</p> </li> <li> <p>
   * <code>kx.sg.4xlarge</code> – The host type with a configuration of 108 GiB
   * memory and 16 vCPUs.</p> </li> <li> <p> <code>kx.sg.8xlarge</code> – The host
   * type with a configuration of 216 GiB memory and 32 vCPUs.</p> </li> <li> <p>
   * <code>kx.sg.16xlarge</code> – The host type with a configuration of 432 GiB
   * memory and 64 vCPUs.</p> </li> <li> <p> <code>kx.sg.32xlarge</code> – The host
   * type with a configuration of 864 GiB memory and 128 vCPUs.</p> </li> <li> <p>
   * <code>kx.sg1.16xlarge</code> – The host type with a configuration of 1949 GiB
   * memory and 64 vCPUs.</p> </li> <li> <p> <code>kx.sg1.24xlarge</code> – The host
   * type with a configuration of 2948 GiB memory and 96 vCPUs.</p> </li> </ul>
   */
  inline const Aws::String& GetHostType() const { return m_hostType; }
  inline bool HostTypeHasBeenSet() const { return m_hostTypeHasBeenSet; }
  template <typename HostTypeT = Aws::String>
  void SetHostType(HostTypeT&& value) {
    m_hostTypeHasBeenSet = true;
    m_hostType = std::forward<HostTypeT>(value);
  }
  template <typename HostTypeT = Aws::String>
  KxScalingGroup& WithHostType(HostTypeT&& value) {
    SetHostType(std::forward<HostTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The list of clusters currently active in a given scaling group. </p>
   */
  inline const Aws::Vector<Aws::String>& GetClusters() const { return m_clusters; }
  inline bool ClustersHasBeenSet() const { return m_clustersHasBeenSet; }
  template <typename ClustersT = Aws::Vector<Aws::String>>
  void SetClusters(ClustersT&& value) {
    m_clustersHasBeenSet = true;
    m_clusters = std::forward<ClustersT>(value);
  }
  template <typename ClustersT = Aws::Vector<Aws::String>>
  KxScalingGroup& WithClusters(ClustersT&& value) {
    SetClusters(std::forward<ClustersT>(value));
    return *this;
  }
  template <typename ClustersT = Aws::String>
  KxScalingGroup& AddClusters(ClustersT&& value) {
    m_clustersHasBeenSet = true;
    m_clusters.emplace_back(std::forward<ClustersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the availability zones.</p>
   */
  inline const Aws::String& GetAvailabilityZoneId() const { return m_availabilityZoneId; }
  inline bool AvailabilityZoneIdHasBeenSet() const { return m_availabilityZoneIdHasBeenSet; }
  template <typename AvailabilityZoneIdT = Aws::String>
  void SetAvailabilityZoneId(AvailabilityZoneIdT&& value) {
    m_availabilityZoneIdHasBeenSet = true;
    m_availabilityZoneId = std::forward<AvailabilityZoneIdT>(value);
  }
  template <typename AvailabilityZoneIdT = Aws::String>
  KxScalingGroup& WithAvailabilityZoneId(AvailabilityZoneIdT&& value) {
    SetAvailabilityZoneId(std::forward<AvailabilityZoneIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The status of scaling groups. </p>
   */
  inline KxScalingGroupStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(KxScalingGroupStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline KxScalingGroup& WithStatus(KxScalingGroupStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The error message when a failed state occurs. </p>
   */
  inline const Aws::String& GetStatusReason() const { return m_statusReason; }
  inline bool StatusReasonHasBeenSet() const { return m_statusReasonHasBeenSet; }
  template <typename StatusReasonT = Aws::String>
  void SetStatusReason(StatusReasonT&& value) {
    m_statusReasonHasBeenSet = true;
    m_statusReason = std::forward<StatusReasonT>(value);
  }
  template <typename StatusReasonT = Aws::String>
  KxScalingGroup& WithStatusReason(StatusReasonT&& value) {
    SetStatusReason(std::forward<StatusReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The last time that the scaling group was updated in FinSpace. The value is
   * determined as epoch time in milliseconds. For example, the value for Monday,
   * November 1, 2021 12:00:00 PM UTC is specified as 1635768000000. </p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTimestamp() const { return m_lastModifiedTimestamp; }
  inline bool LastModifiedTimestampHasBeenSet() const { return m_lastModifiedTimestampHasBeenSet; }
  template <typename LastModifiedTimestampT = Aws::Utils::DateTime>
  void SetLastModifiedTimestamp(LastModifiedTimestampT&& value) {
    m_lastModifiedTimestampHasBeenSet = true;
    m_lastModifiedTimestamp = std::forward<LastModifiedTimestampT>(value);
  }
  template <typename LastModifiedTimestampT = Aws::Utils::DateTime>
  KxScalingGroup& WithLastModifiedTimestamp(LastModifiedTimestampT&& value) {
    SetLastModifiedTimestamp(std::forward<LastModifiedTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The timestamp at which the scaling group was created in FinSpace. The value
   * is determined as epoch time in milliseconds. For example, the value for Monday,
   * November 1, 2021 12:00:00 PM UTC is specified as 1635768000000. </p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTimestamp() const { return m_createdTimestamp; }
  inline bool CreatedTimestampHasBeenSet() const { return m_createdTimestampHasBeenSet; }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  void SetCreatedTimestamp(CreatedTimestampT&& value) {
    m_createdTimestampHasBeenSet = true;
    m_createdTimestamp = std::forward<CreatedTimestampT>(value);
  }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  KxScalingGroup& WithCreatedTimestamp(CreatedTimestampT&& value) {
    SetCreatedTimestamp(std::forward<CreatedTimestampT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_scalingGroupName;
  bool m_scalingGroupNameHasBeenSet = false;

  Aws::String m_hostType;
  bool m_hostTypeHasBeenSet = false;

  Aws::Vector<Aws::String> m_clusters;
  bool m_clustersHasBeenSet = false;

  Aws::String m_availabilityZoneId;
  bool m_availabilityZoneIdHasBeenSet = false;

  KxScalingGroupStatus m_status{KxScalingGroupStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  Aws::String m_statusReason;
  bool m_statusReasonHasBeenSet = false;

  Aws::Utils::DateTime m_lastModifiedTimestamp{};
  bool m_lastModifiedTimestampHasBeenSet = false;

  Aws::Utils::DateTime m_createdTimestamp{};
  bool m_createdTimestampHasBeenSet = false;
};

}  // namespace Model
}  // namespace finspace
}  // namespace Aws
