﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/voice-id/VoiceIDRequest.h>
#include <aws/voice-id/VoiceID_EXPORTS.h>
#include <aws/voice-id/model/InputDataConfig.h>
#include <aws/voice-id/model/OutputDataConfig.h>
#include <aws/voice-id/model/RegistrationConfig.h>

#include <utility>

namespace Aws {
namespace VoiceID {
namespace Model {

/**
 */
class StartFraudsterRegistrationJobRequest : public VoiceIDRequest {
 public:
  AWS_VOICEID_API StartFraudsterRegistrationJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartFraudsterRegistrationJob"; }

  AWS_VOICEID_API Aws::String SerializePayload() const override;

  AWS_VOICEID_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If not provided, the Amazon Web Services SDK
   * populates this field. For more information about idempotency, see <a
   * href="https://aws.amazon.com/builders-library/making-retries-safe-with-idempotent-APIs/">Making
   * retries safe with idempotent APIs</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartFraudsterRegistrationJobRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IAM role Amazon Resource Name (ARN) that grants Voice ID permissions to
   * access customer's buckets to read the input manifest file and write the Job
   * output file. Refer to the <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/voiceid-fraudster-watchlist.html">Create
   * and edit a fraudster watchlist</a> documentation for the permissions needed in
   * this role.</p>
   */
  inline const Aws::String& GetDataAccessRoleArn() const { return m_dataAccessRoleArn; }
  inline bool DataAccessRoleArnHasBeenSet() const { return m_dataAccessRoleArnHasBeenSet; }
  template <typename DataAccessRoleArnT = Aws::String>
  void SetDataAccessRoleArn(DataAccessRoleArnT&& value) {
    m_dataAccessRoleArnHasBeenSet = true;
    m_dataAccessRoleArn = std::forward<DataAccessRoleArnT>(value);
  }
  template <typename DataAccessRoleArnT = Aws::String>
  StartFraudsterRegistrationJobRequest& WithDataAccessRoleArn(DataAccessRoleArnT&& value) {
    SetDataAccessRoleArn(std::forward<DataAccessRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the domain that contains the fraudster registration job and
   * in which the fraudsters are registered.</p>
   */
  inline const Aws::String& GetDomainId() const { return m_domainId; }
  inline bool DomainIdHasBeenSet() const { return m_domainIdHasBeenSet; }
  template <typename DomainIdT = Aws::String>
  void SetDomainId(DomainIdT&& value) {
    m_domainIdHasBeenSet = true;
    m_domainId = std::forward<DomainIdT>(value);
  }
  template <typename DomainIdT = Aws::String>
  StartFraudsterRegistrationJobRequest& WithDomainId(DomainIdT&& value) {
    SetDomainId(std::forward<DomainIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The input data config containing an S3 URI for the input manifest file that
   * contains the list of fraudster registration requests.</p>
   */
  inline const InputDataConfig& GetInputDataConfig() const { return m_inputDataConfig; }
  inline bool InputDataConfigHasBeenSet() const { return m_inputDataConfigHasBeenSet; }
  template <typename InputDataConfigT = InputDataConfig>
  void SetInputDataConfig(InputDataConfigT&& value) {
    m_inputDataConfigHasBeenSet = true;
    m_inputDataConfig = std::forward<InputDataConfigT>(value);
  }
  template <typename InputDataConfigT = InputDataConfig>
  StartFraudsterRegistrationJobRequest& WithInputDataConfig(InputDataConfigT&& value) {
    SetInputDataConfig(std::forward<InputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the new fraudster registration job.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  StartFraudsterRegistrationJobRequest& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The output data config containing the S3 location where Voice ID writes the
   * job output file; you must also include a KMS key ID to encrypt the file.</p>
   */
  inline const OutputDataConfig& GetOutputDataConfig() const { return m_outputDataConfig; }
  inline bool OutputDataConfigHasBeenSet() const { return m_outputDataConfigHasBeenSet; }
  template <typename OutputDataConfigT = OutputDataConfig>
  void SetOutputDataConfig(OutputDataConfigT&& value) {
    m_outputDataConfigHasBeenSet = true;
    m_outputDataConfig = std::forward<OutputDataConfigT>(value);
  }
  template <typename OutputDataConfigT = OutputDataConfig>
  StartFraudsterRegistrationJobRequest& WithOutputDataConfig(OutputDataConfigT&& value) {
    SetOutputDataConfig(std::forward<OutputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The registration config containing details such as the action to take when a
   * duplicate fraudster is detected, and the similarity threshold to use for
   * detecting a duplicate fraudster. </p>
   */
  inline const RegistrationConfig& GetRegistrationConfig() const { return m_registrationConfig; }
  inline bool RegistrationConfigHasBeenSet() const { return m_registrationConfigHasBeenSet; }
  template <typename RegistrationConfigT = RegistrationConfig>
  void SetRegistrationConfig(RegistrationConfigT&& value) {
    m_registrationConfigHasBeenSet = true;
    m_registrationConfig = std::forward<RegistrationConfigT>(value);
  }
  template <typename RegistrationConfigT = RegistrationConfig>
  StartFraudsterRegistrationJobRequest& WithRegistrationConfig(RegistrationConfigT&& value) {
    SetRegistrationConfig(std::forward<RegistrationConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_dataAccessRoleArn;
  bool m_dataAccessRoleArnHasBeenSet = false;

  Aws::String m_domainId;
  bool m_domainIdHasBeenSet = false;

  InputDataConfig m_inputDataConfig;
  bool m_inputDataConfigHasBeenSet = false;

  Aws::String m_jobName;
  bool m_jobNameHasBeenSet = false;

  OutputDataConfig m_outputDataConfig;
  bool m_outputDataConfigHasBeenSet = false;

  RegistrationConfig m_registrationConfig;
  bool m_registrationConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace VoiceID
}  // namespace Aws
