﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/states/SFNRequest.h>
#include <aws/states/SFN_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SFN {
namespace Model {

/**
 */
class RedriveExecutionRequest : public SFNRequest {
 public:
  AWS_SFN_API RedriveExecutionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RedriveExecution"; }

  AWS_SFN_API Aws::String SerializePayload() const override;

  AWS_SFN_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the execution to be redriven.</p>
   */
  inline const Aws::String& GetExecutionArn() const { return m_executionArn; }
  inline bool ExecutionArnHasBeenSet() const { return m_executionArnHasBeenSet; }
  template <typename ExecutionArnT = Aws::String>
  void SetExecutionArn(ExecutionArnT&& value) {
    m_executionArnHasBeenSet = true;
    m_executionArn = std::forward<ExecutionArnT>(value);
  }
  template <typename ExecutionArnT = Aws::String>
  RedriveExecutionRequest& WithExecutionArn(ExecutionArnT&& value) {
    SetExecutionArn(std::forward<ExecutionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If you don’t specify a client token, the Amazon Web
   * Services SDK automatically generates a client token and uses it for the request
   * to ensure idempotency. The API will return idempotent responses for the last 10
   * client tokens used to successfully redrive the execution. These client tokens
   * are valid for up to 15 minutes after they are first used.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  RedriveExecutionRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_executionArn;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_executionArnHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace SFN
}  // namespace Aws
