﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudtrail/CloudTrailRequest.h>
#include <aws/cloudtrail/CloudTrail_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudTrail {
namespace Model {

/**
 */
class GetEventConfigurationRequest : public CloudTrailRequest {
 public:
  AWS_CLOUDTRAIL_API GetEventConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetEventConfiguration"; }

  AWS_CLOUDTRAIL_API Aws::String SerializePayload() const override;

  AWS_CLOUDTRAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the trail for which you want to retrieve event configuration
   * settings.</p>
   */
  inline const Aws::String& GetTrailName() const { return m_trailName; }
  inline bool TrailNameHasBeenSet() const { return m_trailNameHasBeenSet; }
  template <typename TrailNameT = Aws::String>
  void SetTrailName(TrailNameT&& value) {
    m_trailNameHasBeenSet = true;
    m_trailName = std::forward<TrailNameT>(value);
  }
  template <typename TrailNameT = Aws::String>
  GetEventConfigurationRequest& WithTrailName(TrailNameT&& value) {
    SetTrailName(std::forward<TrailNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) or ID suffix of the ARN of the event data
   * store for which you want to retrieve event configuration settings.</p>
   */
  inline const Aws::String& GetEventDataStore() const { return m_eventDataStore; }
  inline bool EventDataStoreHasBeenSet() const { return m_eventDataStoreHasBeenSet; }
  template <typename EventDataStoreT = Aws::String>
  void SetEventDataStore(EventDataStoreT&& value) {
    m_eventDataStoreHasBeenSet = true;
    m_eventDataStore = std::forward<EventDataStoreT>(value);
  }
  template <typename EventDataStoreT = Aws::String>
  GetEventConfigurationRequest& WithEventDataStore(EventDataStoreT&& value) {
    SetEventDataStore(std::forward<EventDataStoreT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_trailName;

  Aws::String m_eventDataStore;
  bool m_trailNameHasBeenSet = false;
  bool m_eventDataStoreHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudTrail
}  // namespace Aws
