﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>
#include <aws/elasticbeanstalk/model/ApplicationResourceLifecycleConfig.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 */
class UpdateApplicationResourceLifecycleRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API UpdateApplicationResourceLifecycleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateApplicationResourceLifecycle"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the application.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  UpdateApplicationResourceLifecycleRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The lifecycle configuration.</p>
   */
  inline const ApplicationResourceLifecycleConfig& GetResourceLifecycleConfig() const { return m_resourceLifecycleConfig; }
  inline bool ResourceLifecycleConfigHasBeenSet() const { return m_resourceLifecycleConfigHasBeenSet; }
  template <typename ResourceLifecycleConfigT = ApplicationResourceLifecycleConfig>
  void SetResourceLifecycleConfig(ResourceLifecycleConfigT&& value) {
    m_resourceLifecycleConfigHasBeenSet = true;
    m_resourceLifecycleConfig = std::forward<ResourceLifecycleConfigT>(value);
  }
  template <typename ResourceLifecycleConfigT = ApplicationResourceLifecycleConfig>
  UpdateApplicationResourceLifecycleRequest& WithResourceLifecycleConfig(ResourceLifecycleConfigT&& value) {
    SetResourceLifecycleConfig(std::forward<ResourceLifecycleConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  ApplicationResourceLifecycleConfig m_resourceLifecycleConfig;
  bool m_applicationNameHasBeenSet = false;
  bool m_resourceLifecycleConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
